library(shiny)

# 定義 UI
ui <- fluidPage(
  titlePanel("Posterior Distribution and UPM Visualization"),
  
  sidebarLayout(
    sidebarPanel(
      selectInput("method", "Select Method:", 
                  choices = c("TPI", "mTPI", "mmTPI")),
      numericInput("x0", "Number of Toxic Events (x0):", value = 3, min = 0),
      numericInput("n0", "Number of Patients (n0):", value = 6, min = 1),
      numericInput("pt", "Target Toxicity Probability (pt):", value = 0.3, min = 0, max = 1),
      numericInput("e10", "Epsilon 1 (e1):", value = 0.05, min = 0),
      numericInput("e20", "Epsilon 2 (e2):", value = 0.05, min = 0),
      actionButton("submit", "Submit")
    ),
    
    mainPanel(
      plotOutput("plot"),
      verbatimTextOutput("results")
    )
  )
)

# 定義 Server
server <- function(input, output) {
  
  calculate <- reactive({
    req(input$submit)
    
    x0 <- input$x0
    n0 <- input$n0
    pt <- input$pt
    e1 <- input$e10
    e2 <- input$e20
    
    alp <- 0.005 + x0
    beta <- 0.005 + n0 - x0
    
    # ========= TPI ========= #
    if (input$method == "TPI") {
      sigma <- sqrt((alp * beta) / (((alp + beta)^2) * (alp + beta + 1)))
      e1 <- min(1.5 * sigma, pt - 0.005)
      e2 <- min(1 * sigma, 1 - pt)
      
      p1 <- pbeta(pt - e1, alp, beta)
      p2 <- pbeta(pt + e2, alp, beta)
      pe <- round(p1, 4)
      ps <- round(p2 - p1, 4)
      pd <- round(1 - p2, 4)
      
      # TPI 判定：面積最大
      areas <- c(LI=pe, EI=ps, HI=pd)
      decision <- names(which.max(areas))
      
      return(list(method="TPI", p=areas, range=c(pt - e1, pt + e2), decision=decision))
    }
    
    # ========= mTPI ========= #
    if (input$method == "mTPI") {
      p1 <- pbeta(pt - e1, alp, beta)
      p2 <- pbeta(pt + e2, alp, beta)
      
      # 計算每個區間的UPM
      upm_ES <- p1 / (pt - e1)
      upm_EI <- (p2 - p1) / (e1 + e2)
      upm_ED <- (1 - p2) / (1 - pt - e2)
      upm_vals <- c(LI=upm_ES, EI=upm_EI, HI=upm_ED)
      
      # 判定：最大UPM
      decision <- names(which.max(upm_vals))
      
      return(list(
        method="mTPI",
        p=upm_vals,
        range=c(pt - e1, pt + e2),
        decision=decision
      ))
    }
    
    # ========= mmTPI ========= #
    if (input$method == "mmTPI") {
      
      L <- pt - e1
      U <- pt + e2
      w <- U - L
      
      intervals <- list(EI = c(L, U))
      
      # 左側 L
      left_end <- L
      i <- 1
      while (left_end > 0) {
        newL <- max(0, left_end - w)
        intervals[[paste0("LI", i)]] <- c(newL, left_end)
        left_end <- newL
        i <- i + 1
      }
      
      # 右側 R
      right_start <- U
      j <- 1
      while (right_start < 1) {
        newR <- min(1, right_start + w)
        intervals[[paste0("HI", j)]] <- c(right_start, newR)
        right_start <- newR
        j <- j + 1
      }
      
      # UPM
      upm <- c()
      for (nm in names(intervals)) {
        a <- intervals[[nm]][1]
        b <- intervals[[nm]][2]
        mass <- pbeta(b, alp, beta) - pbeta(a, alp, beta)
        upm[nm] <- mass / w
      }
      
      # 判定：最大UPM所在區間
      decision <- names(which.max(upm))
      
      return(list(
        method="mmTPI",
        p=upm,
        intervals=intervals,
        decision=decision
      ))
    }
  })
  
  # 結果輸出
  output$results <- renderPrint({
    res <- calculate()
    cat("Method:", res$method, "\n")
    print(res$p)
    
    cat("\nDecision:", res$decision, "\n")
    
    if (res$method == "mmTPI") {
      cat("\nIntervals:\n")
      print(res$intervals)
    } else {
      cat("\nRange:\n")
      print(res$range)
    }
  })
  
  # 圖形
  output$plot <- renderPlot({
    res <- calculate()
    
    alp <- 0.005 + input$x0
    beta <- 0.005 + input$n0 - input$x0
    
    x <- seq(0, 1, length=100)
    y <- dbeta(x, alp, beta)
    
    # Posterior Distribution Curve
    plot(x, y, type="l", lwd=3, col="blue",
         main=paste("Posterior Distribution (", res$method, ")"),
         xlab="Toxicity", ylab="Density",
         xlim=c(0,1))
    
    # ========= mmTPI 背景顏色與 UPM ========= #
    if (res$method == "mmTPI") {
      intervals <- res$intervals
      
      # ES = L*
      L_intervals <- intervals[grep("^LI", names(intervals))]
      for (r in L_intervals) {
        rect(r[1], 0, r[2], max(y), col=rgb(0.8, 1, 0.8, 0.4), border=NA)
      }
      
      # EI
      EI <- intervals$EI
      rect(EI[1], 0, EI[2], max(y), col=rgb(1, 1, 0.6, 0.4), border=NA)
      
      # ED = R*
      R_intervals <- intervals[grep("^HI", names(intervals))]
      for (r in R_intervals) {
        rect(r[1], 0, r[2], max(y), col=rgb(1, 0.8, 0.9, 0.4), border=NA)
      }
      
      # mmTPI 邊界線
      bounds <- unique(sort(unlist(res$intervals)))
      abline(v=bounds, lty=2, lwd=2, col="purple")
      
      # 標示 UPM 橫線
      for (nm in names(res$p)) {
        interval <- intervals[[nm]]
        upm_val <- res$p[[nm]]
        segments(x0=interval[1], y0=upm_val, x1=interval[2], y1=upm_val, col="darkblue", lwd=2)
        text(mean(interval), upm_val, labels=round(upm_val,3), pos=3, cex=0.8)
      }
    }
    
    # ========= mTPI 標示 UPM橫線 ========= #
    if (res$method == "mTPI") {
      p <- res$p
      intervals <- list(
        LI = c(0, res$range[1]),
        EI = c(res$range[1], res$range[2]),
        HI = c(res$range[2], 1)
      )
      for (nm in names(p)) {
        interval <- intervals[[nm]]
        upm_val <- p[[nm]]
        segments(x0=interval[1], y0=upm_val, x1=interval[2], y1=upm_val, col="darkgreen", lwd=2)
        text(mean(interval), upm_val, labels=round(upm_val,3), pos=3, cex=0.8)
      }
    }
    
    # TPI / mTPI 邊界
    if (res$method != "mmTPI") {
      abline(v=res$range, lty=2, lwd=2, col=c("red","green"))
    }
  })
}

shinyApp(ui = ui, server = server)

